import { useState, useEffect } from "react";
import { Link } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Star, Clock, Users, ChevronLeft, ChevronRight } from "lucide-react";
import heroKaaba from "@/assets/hero-kaaba.jpg";
import heroMadinah from "@/assets/hero-madinah.jpg";
import heroTravel from "@/assets/hero-travel.jpg";

const Home = () => {
  const [currentSlide, setCurrentSlide] = useState(0);

  const heroSlides = [
    {
      image: heroKaaba,
      title: "Wujudkan Impian Ibadah Anda",
      subtitle: "Paket Umrah & Haji Terpercaya Sejak 2010"
    },
    {
      image: heroMadinah,
      title: "Nikmati Perjalanan Spiritual",
      subtitle: "Bersama Travel Berpengalaman & Berizin Resmi"
    },
    {
      image: heroTravel,
      title: "Pelayanan Terbaik untuk Jamaah",
      subtitle: "Fasilitas Lengkap & Harga Terjangkau"
    }
  ];

  const featuredPackages = [
    {
      id: 1,
      name: "Paket Umrah Ekonomis",
      description: "Paket hemat dengan fasilitas lengkap dan nyaman",
      duration: "9 Hari",
      price: "Rp 25.000.000",
      quota: 40,
      available: 15,
      status: "Ready Seat",
    },
    {
      id: 2,
      name: "Paket Umrah Platinum",
      description: "Paket premium dengan hotel bintang 5 dekat Masjidil Haram",
      duration: "12 Hari",
      price: "Rp 45.000.000",
      quota: 30,
      available: 8,
      status: "Ready Seat",
    },
    {
      id: 3,
      name: "Paket Umrah Plus Turki",
      description: "Umrah + Wisata Turki dalam satu perjalanan",
      duration: "16 Hari",
      price: "Rp 55.000.000",
      quota: 25,
      available: 0,
      status: "Full Seat",
    },
  ];

  const testimonials = [
    {
      name: "Hj. Siti Aminah",
      rating: 5,
      message: "Alhamdulillah, perjalanan umrah sangat lancar dan nyaman. Pelayanan tour guide sangat membantu. Terima kasih Medina Wisata!",
      photo: "https://api.dicebear.com/7.x/avataaars/svg?seed=Siti"
    },
    {
      name: "H. Ahmad Fauzi",
      rating: 5,
      message: "Hotel dekat dengan Masjidil Haram, makanan enak, dan jadwal ibadah teratur. Sangat puas dengan paket platinum ini.",
      photo: "https://api.dicebear.com/7.x/avataaars/svg?seed=Ahmad"
    },
    {
      name: "Hj. Nur Hasanah",
      rating: 5,
      message: "Ini umrah kedua saya dengan Medina Wisata. Pelayanan tetap memuaskan dan harga terjangkau. Insya Allah umrah lagi tahun depan.",
      photo: "https://api.dicebear.com/7.x/avataaars/svg?seed=Nur"
    }
  ];

  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentSlide((prev) => (prev + 1) % heroSlides.length);
    }, 5000);
    return () => clearInterval(interval);
  }, []);

  const nextSlide = () => setCurrentSlide((prev) => (prev + 1) % heroSlides.length);
  const prevSlide = () => setCurrentSlide((prev) => (prev - 1 + heroSlides.length) % heroSlides.length);

  const getStatusColor = (status: string) => {
    switch (status) {
      case "Ready Seat": return "bg-green-500";
      case "Full Seat": return "bg-red-500";
      case "Sudah Berangkat": return "bg-yellow-500";
      default: return "bg-gray-500";
    }
  };

  return (
    <div className="min-h-screen">
      {/* Hero Carousel */}
      <section className="relative h-[60vh] md:h-[70vh] overflow-hidden">
        {heroSlides.map((slide, index) => (
          <div
            key={index}
            className={`absolute inset-0 transition-opacity duration-1000 ${
              index === currentSlide ? "opacity-100" : "opacity-0"
            }`}
          >
            <img
              src={slide.image}
              alt={slide.title}
              className="w-full h-full object-cover"
            />
            <div className="absolute inset-0 bg-gradient-to-r from-secondary/90 to-primary/70" />
            <div className="absolute inset-0 flex items-center justify-center text-center px-4">
              <div className="max-w-4xl animate-fade-in">
                <h1 className="text-4xl md:text-6xl font-bold text-white mb-4">{slide.title}</h1>
                <p className="text-xl md:text-2xl text-white/90 mb-8">{slide.subtitle}</p>
                <div className="flex gap-4 justify-center flex-wrap">
                  <Link to="/paket">
                    <Button size="lg" className="bg-primary hover:bg-primary/90 text-primary-foreground font-semibold">
                      Lihat Paket
                    </Button>
                  </Link>
                  <Link to="/kontak">
                    <Button size="lg" variant="outline" className="bg-white/10 backdrop-blur-sm text-white border-white hover:bg-white hover:text-secondary">
                      Hubungi Kami
                    </Button>
                  </Link>
                </div>
              </div>
            </div>
          </div>
        ))}
        
        {/* Navigation Arrows */}
        <button
          onClick={prevSlide}
          className="absolute left-4 top-1/2 -translate-y-1/2 bg-white/20 hover:bg-white/30 backdrop-blur-sm text-white p-2 rounded-full transition-all"
        >
          <ChevronLeft className="w-6 h-6" />
        </button>
        <button
          onClick={nextSlide}
          className="absolute right-4 top-1/2 -translate-y-1/2 bg-white/20 hover:bg-white/30 backdrop-blur-sm text-white p-2 rounded-full transition-all"
        >
          <ChevronRight className="w-6 h-6" />
        </button>

        {/* Dots */}
        <div className="absolute bottom-8 left-1/2 -translate-x-1/2 flex gap-2">
          {heroSlides.map((_, index) => (
            <button
              key={index}
              onClick={() => setCurrentSlide(index)}
              className={`w-3 h-3 rounded-full transition-all ${
                index === currentSlide ? "bg-white w-8" : "bg-white/50"
              }`}
            />
          ))}
        </div>
      </section>

      {/* Featured Packages */}
      <section className="container mx-auto px-4 py-16">
        <div className="text-center mb-12 animate-fade-in">
          <h2 className="text-3xl md:text-4xl font-bold text-foreground mb-4">Paket Unggulan</h2>
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            Pilihan paket terbaik untuk perjalanan ibadah Anda dengan fasilitas lengkap dan harga terjangkau
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {featuredPackages.map((pkg) => (
            <Card key={pkg.id} className="overflow-hidden hover:shadow-lg transition-shadow animate-slide-up">
              <div className="h-48 bg-gradient-to-br from-secondary to-primary" />
              <CardHeader>
                <div className="flex justify-between items-start mb-2">
                  <CardTitle className="text-xl">{pkg.name}</CardTitle>
                  <Badge className={`${getStatusColor(pkg.status)} text-white`}>
                    {pkg.status}
                  </Badge>
                </div>
                <CardDescription>{pkg.description}</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-2 text-sm">
                  <div className="flex items-center gap-2">
                    <Clock className="w-4 h-4 text-muted-foreground" />
                    <span>{pkg.duration}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Users className="w-4 h-4 text-muted-foreground" />
                    <span>Tersedia: {pkg.available} dari {pkg.quota} seat</span>
                  </div>
                  <div className="text-2xl font-bold text-primary mt-4">{pkg.price}</div>
                </div>
              </CardContent>
              <CardFooter>
                <Link to={`/paket/${pkg.id}`} className="w-full">
                  <Button className="w-full bg-secondary hover:bg-secondary/90">
                    Lihat Detail
                  </Button>
                </Link>
              </CardFooter>
            </Card>
          ))}
        </div>

        <div className="text-center mt-8">
          <Link to="/paket">
            <Button size="lg" variant="outline">
              Lihat Semua Paket
            </Button>
          </Link>
        </div>
      </section>

      {/* Testimonials */}
      <section className="bg-muted py-16">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12 animate-fade-in">
            <h2 className="text-3xl md:text-4xl font-bold text-foreground mb-4">Testimoni Jamaah</h2>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Pengalaman jamaah yang telah berangkat bersama kami
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            {testimonials.map((testimonial, index) => (
              <Card key={index} className="animate-slide-up">
                <CardHeader>
                  <div className="flex items-center gap-4">
                    <img
                      src={testimonial.photo}
                      alt={testimonial.name}
                      className="w-16 h-16 rounded-full"
                    />
                    <div>
                      <CardTitle className="text-lg">{testimonial.name}</CardTitle>
                      <div className="flex gap-1 mt-1">
                        {[...Array(testimonial.rating)].map((_, i) => (
                          <Star key={i} className="w-4 h-4 fill-yellow-400 text-yellow-400" />
                        ))}
                      </div>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <p className="text-muted-foreground italic">&quot;{testimonial.message}&quot;</p>
                </CardContent>
              </Card>
            ))}
          </div>

          <div className="text-center mt-8">
            <Link to="/testimoni">
              <Button size="lg" variant="outline">
                Lihat Semua Testimoni
              </Button>
            </Link>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="container mx-auto px-4 py-16">
        <div className="bg-gradient-to-r from-secondary to-primary rounded-2xl p-8 md:p-12 text-center text-white">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">Siap Berangkat Umrah?</h2>
          <p className="text-lg md:text-xl mb-8 text-white/90">
            Konsultasikan kebutuhan ibadah Anda bersama tim ahli kami
          </p>
          <div className="flex gap-4 justify-center flex-wrap">
            <Link to="/jadwal">
              <Button size="lg" className="bg-white text-secondary hover:bg-white/90">
                Cek Jadwal Keberangkatan
              </Button>
            </Link>
            <Link to="/kontak">
              <Button size="lg" variant="outline" className="bg-white/10 backdrop-blur-sm text-white border-white hover:bg-white hover:text-secondary">
                Konsultasi Gratis
              </Button>
            </Link>
          </div>
        </div>
      </section>
    </div>
  );
};

export default Home;
