import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Calendar, Users, Plane, MessageCircle } from "lucide-react";
import { getJadwalMessage, openWhatsApp } from "@/lib/whatsapp";

const Jadwal = () => {
  const jadwalData = [
    {
      id: 1,
      tanggal: "15 Maret 2025",
      paket: "Paket Umrah Ekonomis",
      harga: "Rp 25.000.000",
      quota: 40,
      tersedia: 15,
      status: "Ready Seat"
    },
    {
      id: 2,
      tanggal: "22 Maret 2025",
      paket: "Paket Umrah Platinum",
      harga: "Rp 45.000.000",
      quota: 30,
      tersedia: 8,
      status: "Ready Seat"
    },
    {
      id: 3,
      tanggal: "1 April 2025",
      paket: "Paket Umrah Reguler",
      harga: "Rp 28.000.000",
      quota: 45,
      tersedia: 30,
      status: "Ready Seat"
    },
    {
      id: 4,
      tanggal: "10 April 2025",
      paket: "Paket Umrah Plus Turki",
      harga: "Rp 55.000.000",
      quota: 25,
      tersedia: 0,
      status: "Full Seat"
    },
    {
      id: 5,
      tanggal: "20 April 2025",
      paket: "Paket Umrah VIP",
      harga: "Rp 65.000.000",
      quota: 20,
      tersedia: 12,
      status: "Ready Seat"
    },
    {
      id: 6,
      tanggal: "1 Mei 2025",
      paket: "Paket Umrah Ramadhan",
      harga: "Rp 52.000.000",
      quota: 35,
      tersedia: 0,
      status: "Sudah Berangkat"
    }
  ];

  const getStatusColor = (status: string) => {
    switch (status) {
      case "Ready Seat": return "bg-green-500 hover:bg-green-600";
      case "Full Seat": return "bg-red-500 hover:bg-red-600";
      case "Sudah Berangkat": return "bg-yellow-500 hover:bg-yellow-600";
      default: return "bg-gray-500 hover:bg-gray-600";
    }
  };

  const handleRegister = (item: any) => {
    const message = getJadwalMessage(item.tanggal, item.paket, item.harga, item.tersedia);
    openWhatsApp(message);
  };

  return (
    <div className="min-h-screen py-12">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-12 animate-fade-in">
          <h1 className="text-4xl md:text-5xl font-bold text-foreground mb-4">Jadwal Keberangkatan</h1>
          <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
            Pilih jadwal keberangkatan yang sesuai dengan rencana Anda. Daftar segera sebelum kuota penuh!
          </p>
        </div>

        {/* Desktop Table View */}
        <div className="hidden md:block overflow-x-auto animate-slide-up">
          <table className="w-full border-collapse">
            <thead>
              <tr className="bg-secondary text-secondary-foreground">
                <th className="p-4 text-left font-semibold">Tanggal Keberangkatan</th>
                <th className="p-4 text-left font-semibold">Paket</th>
                <th className="p-4 text-left font-semibold">Harga</th>
                <th className="p-4 text-left font-semibold">Kuota</th>
                <th className="p-4 text-center font-semibold">Status</th>
                <th className="p-4 text-center font-semibold">Aksi</th>
              </tr>
            </thead>
            <tbody>
              {jadwalData.map((item, index) => (
                <tr key={item.id} className={index % 2 === 0 ? "bg-muted/50" : "bg-background"}>
                  <td className="p-4">
                    <div className="flex items-center gap-2">
                      <Calendar className="w-5 h-5 text-primary" />
                      <span className="font-medium">{item.tanggal}</span>
                    </div>
                  </td>
                  <td className="p-4">
                    <div className="flex items-center gap-2">
                      <Plane className="w-5 h-5 text-muted-foreground" />
                      <span>{item.paket}</span>
                    </div>
                  </td>
                  <td className="p-4 font-semibold text-primary">{item.harga}</td>
                  <td className="p-4">
                    <div className="flex items-center gap-2">
                      <Users className="w-5 h-5 text-muted-foreground" />
                      <span>{item.tersedia} / {item.quota}</span>
                    </div>
                  </td>
                  <td className="p-4 text-center">
                    <Badge className={`${getStatusColor(item.status)} text-white`}>
                      {item.status}
                    </Badge>
                  </td>
                  <td className="p-4 text-center">
                    {item.status === "Ready Seat" ? (
                      <Button 
                        size="sm"
                        className="bg-[#25D366] hover:bg-[#20BA5A] text-white"
                        onClick={() => handleRegister(item)}
                      >
                        <MessageCircle className="w-4 h-4 mr-1" />
                        Daftar
                      </Button>
                    ) : (
                      <Button size="sm" variant="outline" disabled>
                        {item.status === "Full Seat" ? "Penuh" : "Berangkat"}
                      </Button>
                    )}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Mobile Card View */}
        <div className="md:hidden space-y-4 animate-slide-up">
          {jadwalData.map((item) => (
            <Card key={item.id}>
              <CardContent className="p-6">
                <div className="flex justify-between items-start mb-4">
                  <div>
                    <div className="flex items-center gap-2 mb-1">
                      <Calendar className="w-5 h-5 text-primary" />
                      <span className="font-bold">{item.tanggal}</span>
                    </div>
                    <div className="flex items-center gap-2 text-muted-foreground">
                      <Plane className="w-4 h-4" />
                      <span className="text-sm">{item.paket}</span>
                    </div>
                  </div>
                  <Badge className={`${getStatusColor(item.status)} text-white`}>
                    {item.status}
                  </Badge>
                </div>

                <div className="space-y-2 mb-4">
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">Harga:</span>
                    <span className="font-semibold text-primary">{item.harga}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">Kuota:</span>
                    <span className="flex items-center gap-1">
                      <Users className="w-4 h-4" />
                      {item.tersedia} / {item.quota}
                    </span>
                  </div>
                </div>

                {item.status === "Ready Seat" ? (
                  <Button 
                    className="w-full bg-[#25D366] hover:bg-[#20BA5A] text-white"
                    onClick={() => handleRegister(item)}
                  >
                    <MessageCircle className="w-4 h-4 mr-2" />
                    Daftar Sekarang
                  </Button>
                ) : (
                  <Button className="w-full" variant="outline" disabled>
                    {item.status === "Full Seat" ? "Kuota Penuh" : "Sudah Berangkat"}
                  </Button>
                )}
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Info Section */}
        <div className="mt-12 bg-muted rounded-xl p-8 animate-fade-in">
          <h2 className="text-2xl font-bold text-foreground mb-4 text-center">Informasi Penting</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
            <div>
              <h3 className="font-bold mb-2 text-lg">✅ Cara Mendaftar:</h3>
              <ul className="text-sm text-muted-foreground space-y-1 list-disc list-inside">
                <li>Klik tombol "Daftar" pada jadwal yang dipilih</li>
                <li>Anda akan diarahkan ke WhatsApp kami</li>
                <li>Tim kami akan membantu proses pendaftaran</li>
                <li>Siapkan dokumen: KTP, KK, Foto 4x6</li>
              </ul>
            </div>
            <div>
              <h3 className="font-bold mb-2 text-lg">💰 Sistem Pembayaran:</h3>
              <ul className="text-sm text-muted-foreground space-y-1 list-disc list-inside">
                <li>DP minimal 30% untuk booking seat</li>
                <li>Cicilan tersedia hingga H-60 hari</li>
                <li>Pelunasan H-30 hari sebelum keberangkatan</li>
                <li>Transfer ke rekening resmi PT Medina Wisata</li>
              </ul>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Jadwal;
