import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { MapPin, Phone, Mail, Clock, MessageCircle, Facebook, Instagram, Youtube } from "lucide-react";
import { WA_NUMBER, TELP, EMAIL, openWhatsApp } from "@/lib/whatsapp";
import { toast } from "sonner";

const Kontak = () => {
  const [formData, setFormData] = useState({
    nama: "",
    email: "",
    pesan: ""
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.nama || !formData.email || !formData.pesan) {
      toast.error("Mohon lengkapi semua field");
      return;
    }

    // In real app, save to database
    toast.success("Pesan Anda telah terkirim! Kami akan segera menghubungi Anda.");
    
    // Reset form
    setFormData({ nama: "", email: "", pesan: "" });
  };

  const handleWhatsApp = () => {
    const message = "Halo, saya ingin bertanya tentang paket umrah dan haji. Terima kasih.";
    openWhatsApp(message);
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    });
  };

  return (
    <div className="min-h-screen py-12">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-12 animate-fade-in">
          <h1 className="text-4xl md:text-5xl font-bold text-foreground mb-4">Hubungi Kami</h1>
          <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
            Kami siap membantu menjawab pertanyaan dan konsultasi Anda seputar paket umrah dan haji
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-12">
          {/* Contact Info Cards */}
          <Card className="hover:shadow-lg transition-all animate-slide-up">
            <CardContent className="p-6 text-center">
              <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                <Phone className="w-8 h-8 text-primary" />
              </div>
              <h3 className="font-bold text-lg mb-2">Telepon</h3>
              <a href={`tel:${TELP.replace(/-/g, '')}`} className="text-muted-foreground hover:text-primary transition-colors">
                {TELP}
              </a>
              <p className="text-sm text-muted-foreground mt-2">Senin - Sabtu, 08:00 - 17:00 WIB</p>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-all animate-slide-up">
            <CardContent className="p-6 text-center">
              <div className="w-16 h-16 bg-[#25D366]/10 rounded-full flex items-center justify-center mx-auto mb-4">
                <MessageCircle className="w-8 h-8 text-[#25D366]" />
              </div>
              <h3 className="font-bold text-lg mb-2">WhatsApp</h3>
              <button onClick={handleWhatsApp} className="text-muted-foreground hover:text-[#25D366] transition-colors">
                {TELP}
              </button>
              <p className="text-sm text-muted-foreground mt-2">Fast Response 24/7</p>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-all animate-slide-up">
            <CardContent className="p-6 text-center">
              <div className="w-16 h-16 bg-secondary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                <Mail className="w-8 h-8 text-secondary" />
              </div>
              <h3 className="font-bold text-lg mb-2">Email</h3>
              <a href={`mailto:${EMAIL}`} className="text-muted-foreground hover:text-secondary transition-colors break-all">
                {EMAIL}
              </a>
              <p className="text-sm text-muted-foreground mt-2">Respon dalam 24 jam</p>
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Contact Form */}
          <Card className="animate-slide-up">
            <CardHeader>
              <CardTitle className="text-2xl">Kirim Pesan</CardTitle>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <Label htmlFor="nama">Nama Lengkap *</Label>
                  <Input
                    id="nama"
                    name="nama"
                    placeholder="Masukkan nama lengkap"
                    value={formData.nama}
                    onChange={handleChange}
                    className="mt-1"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="email">Email *</Label>
                  <Input
                    id="email"
                    name="email"
                    type="email"
                    placeholder="nama@email.com"
                    value={formData.email}
                    onChange={handleChange}
                    className="mt-1"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="pesan">Pesan *</Label>
                  <Textarea
                    id="pesan"
                    name="pesan"
                    placeholder="Tuliskan pesan atau pertanyaan Anda..."
                    value={formData.pesan}
                    onChange={handleChange}
                    className="mt-1 min-h-[150px]"
                    required
                  />
                </div>

                <Button type="submit" size="lg" className="w-full bg-secondary hover:bg-secondary/90">
                  Kirim Pesan
                </Button>
              </form>

              <div className="mt-6 pt-6 border-t">
                <p className="text-sm text-muted-foreground mb-4 text-center">
                  Atau hubungi kami langsung via WhatsApp untuk respons lebih cepat
                </p>
                <Button 
                  onClick={handleWhatsApp}
                  size="lg" 
                  className="w-full bg-[#25D366] hover:bg-[#20BA5A] text-white"
                >
                  <MessageCircle className="w-5 h-5 mr-2" />
                  Chat via WhatsApp
                </Button>
              </div>
            </CardContent>
          </Card>

          {/* Map & Office Info */}
          <div className="space-y-6 animate-slide-up">
            <Card>
              <CardHeader>
                <CardTitle className="text-2xl">Kantor Kami</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-start gap-3">
                  <MapPin className="w-5 h-5 text-primary flex-shrink-0 mt-1" />
                  <div>
                    <h4 className="font-semibold mb-1">Alamat</h4>
                    <p className="text-muted-foreground">
                      Tasikmalaya, Jawa Barat, Indonesia
                    </p>
                  </div>
                </div>

                <div className="flex items-start gap-3">
                  <Clock className="w-5 h-5 text-primary flex-shrink-0 mt-1" />
                  <div>
                    <h4 className="font-semibold mb-1">Jam Operasional</h4>
                    <p className="text-muted-foreground">Senin - Jumat: 08:00 - 17:00 WIB</p>
                    <p className="text-muted-foreground">Sabtu: 08:00 - 14:00 WIB</p>
                    <p className="text-muted-foreground">Minggu & Tanggal Merah: Tutup</p>
                  </div>
                </div>

                <div className="pt-4 border-t">
                  <h4 className="font-semibold mb-3">Follow Us</h4>
                  <div className="flex gap-3">
                    <a href="#" className="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center text-white hover:bg-blue-700 transition-colors">
                      <Facebook className="w-5 h-5" />
                    </a>
                    <a href="#" className="w-10 h-10 bg-pink-600 rounded-full flex items-center justify-center text-white hover:bg-pink-700 transition-colors">
                      <Instagram className="w-5 h-5" />
                    </a>
                    <a href="#" className="w-10 h-10 bg-red-600 rounded-full flex items-center justify-center text-white hover:bg-red-700 transition-colors">
                      <Youtube className="w-5 h-5" />
                    </a>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Google Maps Embed */}
            <Card className="overflow-hidden">
              <iframe
                src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d63307.557966576416!2d108.49282795293956!3d-7.384976990564066!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x61804d61dde58f9f%3A0x9e61f1a43db23aa2!2sPT.%20Medina%20Wisata%20Haramain!5e0!3m2!1sid!2sid!4v1761408274957!5m2!1sid!2sid"
                width="100%"
                height="300"
                style={{ border: 0 }}
                allowFullScreen
                loading="lazy"
                referrerPolicy="no-referrer-when-downgrade"
                title="Lokasi PT Medina Wisata Haramain"
              />
            </Card>
          </div>
        </div>

        {/* FAQ Section */}
        <Card className="mt-12 bg-muted animate-fade-in">
          <CardContent className="p-8">
            <h2 className="text-2xl font-bold text-center mb-8">Pertanyaan yang Sering Diajukan</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <h3 className="font-bold mb-2">Bagaimana cara mendaftar umrah?</h3>
                <p className="text-sm text-muted-foreground">
                  Anda bisa menghubungi kami via WhatsApp atau datang langsung ke kantor. 
                  Siapkan KTP, KK, dan foto 4x6 untuk proses pendaftaran.
                </p>
              </div>
              <div>
                <h3 className="font-bold mb-2">Apakah bisa cicilan?</h3>
                <p className="text-sm text-muted-foreground">
                  Ya, kami menyediakan sistem cicilan hingga H-60 hari keberangkatan. 
                  DP minimal 30% untuk booking seat.
                </p>
              </div>
              <div>
                <h3 className="font-bold mb-2">Berapa lama proses visa?</h3>
                <p className="text-sm text-muted-foreground">
                  Proses visa umrah memakan waktu sekitar 14-30 hari kerja. 
                  Kami akan membantu mengurus semua keperluan visa Anda.
                </p>
              </div>
              <div>
                <h3 className="font-bold mb-2">Apa saja yang termasuk dalam paket?</h3>
                <p className="text-sm text-muted-foreground">
                  Semua paket sudah termasuk tiket, hotel, makan, visa, transportasi, 
                  tour guide, dan perlengkapan umrah.
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default Kontak;
