import { useParams, Link } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent } from "@/components/ui/card";
import { Clock, Users, Hotel, Plane, MapPin, Check, MessageCircle } from "lucide-react";
import { getBookingMessage, getConsultationMessage, openWhatsApp } from "@/lib/whatsapp";

const PaketDetail = () => {
  const { id } = useParams();

  // Mock data - in real app, fetch from database
  const paketData: any = {
    "1": {
      name: "Paket Umrah Ekonomis",
      description: "Paket hemat dengan fasilitas lengkap dan nyaman untuk perjalanan ibadah Anda. Cocok untuk jamaah yang menginginkan pengalaman umrah berkualitas dengan budget terjangkau.",
      duration: "9 Hari 8 Malam",
      price: "Rp 25.000.000",
      quota: 40,
      available: 15,
      status: "Ready Seat",
      hotel: {
        makkah: "Hotel Bintang 3 (500m dari Masjidil Haram)",
        madinah: "Hotel Bintang 3 (300m dari Masjid Nabawi)"
      },
      features: [
        "Tiket pesawat PP (Jakarta - Jeddah)",
        "Hotel bintang 3 di Makkah & Madinah",
        "Makan 3x sehari (breakfast, lunch, dinner)",
        "Visa umrah & asuransi perjalanan",
        "Tour guide berpengalaman berbahasa Indonesia",
        "Perlengkapan umrah (koper, tas, mukena)",
        "City tour Makkah & Madinah",
        "Ziarah ke tempat bersejarah",
        "Transportasi bus AC selama di Arab Saudi",
        "Manasik umrah sebelum keberangkatan"
      ],
      itinerary: [
        { day: "Hari 1-2", activity: "Keberangkatan Jakarta - Jeddah - Makkah, check in hotel, istirahat" },
        { day: "Hari 3-5", activity: "Ibadah umrah, tawaf, sa'i, ziarah Makkah" },
        { day: "Hari 6", activity: "Perjalanan Makkah - Madinah" },
        { day: "Hari 7-8", activity: "Ziarah Madinah, shalat di Masjid Nabawi" },
        { day: "Hari 9", activity: "Check out, perjalanan ke Jeddah, penerbangan pulang" }
      ]
    },
    "2": {
      name: "Paket Umrah Platinum",
      description: "Paket premium dengan hotel bintang 5 dekat Masjidil Haram dan Masjid Nabawi. Pengalaman umrah yang sangat nyaman dengan fasilitas terbaik.",
      duration: "12 Hari 11 Malam",
      price: "Rp 45.000.000",
      quota: 30,
      available: 8,
      status: "Ready Seat",
      hotel: {
        makkah: "Hotel Bintang 5 (100m dari Masjidil Haram)",
        madinah: "Hotel Bintang 5 (100m dari Masjid Nabawi)"
      },
      features: [
        "Tiket pesawat PP kelas ekonomi premium",
        "Hotel bintang 5 dengan view Ka'bah",
        "Buffet premium 3x sehari",
        "Visa umrah & asuransi premium",
        "Tour guide profesional & muthawif resmi",
        "Perlengkapan umrah premium (koper fiber, dll)",
        "City tour eksklusif Makkah & Madinah",
        "Ziarah lengkap + Jabal Rahmah",
        "Bus AC VIP selama di Arab Saudi",
        "Airport lounge access",
        "Welcome dinner & farewell dinner",
        "Oleh-oleh khas Arab Saudi"
      ],
      itinerary: [
        { day: "Hari 1-2", activity: "Keberangkatan dengan airport lounge, tiba Jeddah, check in hotel bintang 5" },
        { day: "Hari 3-7", activity: "Ibadah umrah berkali-kali, tawaf, sa'i, ziarah Makkah lengkap" },
        { day: "Hari 8", activity: "Perjalanan Makkah - Madinah dengan bus VIP" },
        { day: "Hari 9-11", activity: "Shalat 40 waktu di Masjid Nabawi, ziarah Madinah" },
        { day: "Hari 12", activity: "Check out, shopping, penerbangan pulang" }
      ]
    }
  };

  const paket = paketData[id as string] || paketData["1"];

  const handleBooking = () => {
    const message = getBookingMessage(paket.name, paket.price, paket.duration, paket.available);
    openWhatsApp(message);
  };

  const handleConsultation = () => {
    const message = getConsultationMessage(paket.name);
    openWhatsApp(message);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "Ready Seat": return "bg-green-500 hover:bg-green-600";
      case "Full Seat": return "bg-red-500 hover:bg-red-600";
      case "Sudah Berangkat": return "bg-yellow-500 hover:bg-yellow-600";
      default: return "bg-gray-500 hover:bg-gray-600";
    }
  };

  return (
    <div className="min-h-screen py-12">
      <div className="container mx-auto px-4">
        {/* Breadcrumb */}
        <div className="mb-6 text-sm text-muted-foreground">
          <Link to="/" className="hover:text-foreground">Beranda</Link>
          {" / "}
          <Link to="/paket" className="hover:text-foreground">Paket</Link>
          {" / "}
          <span className="text-foreground">{paket.name}</span>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Left: Image */}
          <div className="animate-fade-in">
            <div className="h-[400px] bg-gradient-to-br from-secondary via-primary/80 to-accent rounded-xl overflow-hidden relative">
              <div className="absolute inset-0 flex items-center justify-center">
                <div className="text-white text-center">
                  <Plane className="w-24 h-24 mx-auto mb-4 opacity-80" />
                  <Hotel className="w-16 h-16 mx-auto opacity-60" />
                </div>
              </div>
              <Badge className={`absolute top-4 right-4 ${getStatusColor(paket.status)} text-white text-lg px-4 py-2`}>
                {paket.status}
              </Badge>
            </div>

            {/* Hotel Info Cards */}
            <div className="mt-4 space-y-3">
              <Card>
                <CardContent className="p-4 flex items-start gap-3">
                  <Hotel className="w-5 h-5 text-primary flex-shrink-0 mt-0.5" />
                  <div>
                    <p className="font-semibold text-sm">Hotel Makkah</p>
                    <p className="text-sm text-muted-foreground">{paket.hotel.makkah}</p>
                  </div>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4 flex items-start gap-3">
                  <Hotel className="w-5 h-5 text-secondary flex-shrink-0 mt-0.5" />
                  <div>
                    <p className="font-semibold text-sm">Hotel Madinah</p>
                    <p className="text-sm text-muted-foreground">{paket.hotel.madinah}</p>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>

          {/* Right: Info */}
          <div className="animate-slide-up">
            <h1 className="text-3xl md:text-4xl font-bold text-foreground mb-4">{paket.name}</h1>
            <p className="text-muted-foreground mb-6">{paket.description}</p>

            <div className="space-y-3 mb-6">
              <div className="flex items-center gap-3">
                <Clock className="w-5 h-5 text-muted-foreground" />
                <span className="text-lg">{paket.duration}</span>
              </div>
              <div className="flex items-center gap-3">
                <Users className="w-5 h-5 text-muted-foreground" />
                <span className="text-lg">Tersedia: {paket.available} dari {paket.quota} seat</span>
              </div>
            </div>

            <div className="bg-muted rounded-lg p-6 mb-6">
              <div className="text-sm text-muted-foreground mb-2">Harga Paket</div>
              <div className="text-4xl font-bold text-primary mb-4">{paket.price}</div>
              <div className="text-sm text-muted-foreground">*Harga per orang, sudah termasuk semua fasilitas</div>
            </div>

            {/* WhatsApp Buttons */}
            <div className="space-y-3">
              <Button 
                size="lg" 
                className="w-full bg-[#25D366] hover:bg-[#20BA5A] text-white text-lg py-6"
                onClick={handleBooking}
              >
                <MessageCircle className="mr-2 h-5 w-5" />
                Booking Sekarang via WhatsApp
              </Button>
              <Button 
                size="lg" 
                variant="outline" 
                className="w-full text-lg py-6"
                onClick={handleConsultation}
              >
                <MessageCircle className="mr-2 h-5 w-5" />
                Konsultasi Paket
              </Button>
            </div>
          </div>
        </div>

        {/* Facilities */}
        <div className="mt-12">
          <h2 className="text-2xl font-bold mb-6">Fasilitas Lengkap</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {paket.features.map((feature: string, index: number) => (
              <div key={index} className="flex items-start gap-3 p-4 bg-muted rounded-lg">
                <Check className="w-5 h-5 text-green-600 flex-shrink-0 mt-0.5" />
                <span>{feature}</span>
              </div>
            ))}
          </div>
        </div>

        {/* Itinerary */}
        <div className="mt-12">
          <h2 className="text-2xl font-bold mb-6">Rencana Perjalanan</h2>
          <div className="space-y-4">
            {paket.itinerary.map((item: any, index: number) => (
              <Card key={index}>
                <CardContent className="p-6">
                  <div className="flex items-start gap-4">
                    <div className="bg-primary text-primary-foreground rounded-full w-12 h-12 flex items-center justify-center font-bold flex-shrink-0">
                      {index + 1}
                    </div>
                    <div>
                      <h3 className="font-bold text-lg mb-1">{item.day}</h3>
                      <p className="text-muted-foreground">{item.activity}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* CTA Bottom */}
        <div className="mt-12 bg-gradient-to-r from-secondary to-primary rounded-xl p-8 text-center text-white">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Siap Berangkat?</h2>
          <p className="text-lg mb-6 text-white/90">Segera daftarkan diri Anda sebelum kuota habis!</p>
          <div className="flex gap-4 justify-center flex-wrap">
            <Button 
              size="lg" 
              className="bg-white text-secondary hover:bg-white/90"
              onClick={handleBooking}
            >
              <MessageCircle className="mr-2 h-5 w-5" />
              Booking via WhatsApp
            </Button>
            <Link to="/jadwal">
              <Button size="lg" variant="outline" className="bg-white/10 backdrop-blur-sm text-white border-white hover:bg-white hover:text-secondary">
                Lihat Jadwal Keberangkatan
              </Button>
            </Link>
          </div>
        </div>
      </div>
    </div>
  );
};

export default PaketDetail;
