import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { MessageCircle, Heart } from "lucide-react";
import { getTitipDoaMessage, openWhatsApp } from "@/lib/whatsapp";
import { toast } from "sonner";

const TitipDoa = () => {
  const [formData, setFormData] = useState({
    nama: "",
    alamat: "",
    doa: ""
  });

  // Mock data for approved prayers
  const [prayers, setPrayers] = useState([
    {
      id: 1,
      nama: "Ibu Fatimah",
      alamat: "Jakarta",
      doa: "Ya Allah, mudahkan rezeki keluarga kami dan lindungi kami dari segala penyakit",
      date: "2 hari yang lalu"
    },
    {
      id: 2,
      nama: "Bapak Hasan",
      alamat: "Bandung",
      doa: "Ya Rabb, karuniakan kesehatan untuk ibu saya dan panjangkan umurnya",
      date: "3 hari yang lalu"
    },
    {
      id: 3,
      nama: "Saudara Ahmad",
      alamat: "Surabaya",
      doa: "Allahumma berikanlah kemudahan dalam pekerjaan dan keberkahan dalam keluarga kami",
      date: "5 hari yang lalu"
    }
  ]);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.nama || !formData.alamat || !formData.doa) {
      toast.error("Mohon lengkapi semua field");
      return;
    }

    // Add to list with animation
    const newPrayer = {
      id: prayers.length + 1,
      nama: formData.nama,
      alamat: formData.alamat,
      doa: formData.doa,
      date: "Baru saja"
    };
    
    setPrayers([newPrayer, ...prayers]);

    // Open WhatsApp
    const message = getTitipDoaMessage(formData.nama, formData.alamat, formData.doa);
    openWhatsApp(message);

    // Reset form
    setFormData({ nama: "", alamat: "", doa: "" });
    
    toast.success("Doa Anda telah dikirim! Akan kami doakan di tanah suci.");
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    });
  };

  return (
    <div className="min-h-screen py-12">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-12 animate-fade-in">
          <h1 className="text-4xl md:text-5xl font-bold text-foreground mb-4">Titip Doa</h1>
          <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
            Titipkan doa Anda untuk didoakan di tanah suci Makkah dan Madinah. Semoga Allah mengabulkan setiap doa kita.
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Left: Form */}
          <Card className="animate-slide-up">
            <CardHeader>
              <CardTitle className="text-2xl">Kirim Doa Anda</CardTitle>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <Label htmlFor="nama">Nama Lengkap *</Label>
                  <Input
                    id="nama"
                    name="nama"
                    placeholder="Masukkan nama lengkap"
                    value={formData.nama}
                    onChange={handleChange}
                    className="mt-1"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="alamat">Alamat *</Label>
                  <Input
                    id="alamat"
                    name="alamat"
                    placeholder="Kota/Kabupaten"
                    value={formData.alamat}
                    onChange={handleChange}
                    className="mt-1"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="doa">Doa yang Ingin Dititipkan *</Label>
                  <Textarea
                    id="doa"
                    name="doa"
                    placeholder="Tuliskan doa Anda di sini..."
                    value={formData.doa}
                    onChange={handleChange}
                    className="mt-1 min-h-[120px]"
                    required
                  />
                  <p className="text-sm text-muted-foreground mt-1">
                    Tuliskan doa dengan penuh keikhlasan dan keyakinan
                  </p>
                </div>

                <Button 
                  type="submit"
                  size="lg"
                  className="w-full bg-[#25D366] hover:bg-[#20BA5A] text-white"
                >
                  <MessageCircle className="w-5 h-5 mr-2" />
                  Kirim Doa via WhatsApp
                </Button>

                <p className="text-xs text-center text-muted-foreground">
                  Setelah mengirim, Anda akan diarahkan ke WhatsApp untuk konfirmasi. Doa akan kami bawa ke tanah suci.
                </p>
              </form>
            </CardContent>
          </Card>

          {/* Right: Prayer List */}
          <div className="animate-slide-up">
            <Card className="mb-6">
              <CardHeader>
                <CardTitle className="text-2xl flex items-center gap-2">
                  <Heart className="w-6 h-6 text-red-500" />
                  Doa-doa Jamaah
                </CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground mb-4">
                  Berikut adalah doa-doa dari saudara-saudara kita yang telah kami terima. Mari kita ikut mendoakan bersama.
                </p>
                
                <div className="space-y-4 max-h-[600px] overflow-y-auto pr-2">
                  {prayers.map((prayer) => (
                    <div 
                      key={prayer.id} 
                      className="p-4 border rounded-lg bg-muted/50 hover:bg-muted transition-colors animate-fade-in"
                    >
                      <div className="flex justify-between items-start mb-2">
                        <div>
                          <h4 className="font-semibold text-foreground">{prayer.nama}</h4>
                          <p className="text-sm text-muted-foreground">{prayer.alamat}</p>
                        </div>
                        <span className="text-xs text-muted-foreground">{prayer.date}</span>
                      </div>
                      <p className="text-sm italic text-muted-foreground mt-2 leading-relaxed">
                        &quot;{prayer.doa}&quot;
                      </p>
                      <div className="mt-3 flex items-center gap-2 text-xs text-muted-foreground">
                        <Heart className="w-4 h-4 text-red-500" />
                        <span>Aamiin Ya Rabbal Alamin</span>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Info Card */}
            <Card className="bg-gradient-to-br from-secondary to-primary text-white">
              <CardContent className="p-6">
                <h3 className="font-bold text-lg mb-2">📿 Layanan Titip Doa</h3>
                <ul className="text-sm space-y-2 text-white/90">
                  <li>✅ Doa akan dibacakan di Masjidil Haram</li>
                  <li>✅ Doa akan dibacakan di Masjid Nabawi</li>
                  <li>✅ Gratis, tidak dipungut biaya</li>
                  <li>✅ Privasi terjaga, nama bisa disamarkan</li>
                </ul>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Hadits Section */}
        <Card className="mt-12 bg-muted animate-fade-in">
          <CardContent className="p-8 text-center">
            <p className="text-lg italic text-foreground mb-2">
              &quot;Doa orang yang berdoa untuk saudaranya yang tidak berada di hadapannya, akan dikabulkan.&quot;
            </p>
            <p className="text-sm text-muted-foreground">
              (HR. Muslim)
            </p>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};

export default TitipDoa;
